/*
  Copyright (c) 2003 Stefanos Harhalakis

  This file is part of netmap.

  netmap is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  netmap is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with netmap; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "snmp.h"

#include "db.h"

#include "v/misc/keytable2.cc"
#include "v/misc/nkeytable.cc"

#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#define	GET_IFDESCR_MIB		"IF-MIB::ifDescr"
#define GET_IFINOCTETS_MIB	"IF-MIB::ifInOctets"
#define GET_IFOUTOCTETS_MIB	"IF-MIB::ifOutOctets"
#define GET_IFSPEED_MIB		"IF-MIB::ifSpeed"
#define GET_IFALIAS_MIB		"IF-MIB::ifAlias"

#define	GET_SYSUPTIME0		"SNMPv2-MIB::sysUpTime.0"

#define	GET_CISCO_CPU_5S	"SNMPv2-SMI::enterprises.9.2.1.56.0"
#define	GET_CISCO_CPU_1M	"SNMPv2-SMI::enterprises.9.2.1.57.0"
#define	GET_CISCO_CPU_5M	"SNMPv2-SMI::enterprises.9.2.1.58.0"
#define GET_CISCO_MEMUSED	"SNMPv2-SMI::enterprises.9.9.48.1.1.1.5.1"
#define GET_CISCO_MEMFREE	"SNMPv2-SMI::enterprises.9.9.48.1.1.1.6.1"

#define	FL_VENDOR		0x0f
#define	FL_VENDOR_OTHER		0x00
#define	FL_VENDOR_CISCO		0x01
#define FL_VENDOR_NETSNMP	0x02
#define	FL_VENDOR_ALL		0x0f

#define IS_OTHER(x)		( (x & FL_VENDOR) == FL_VENDOR_OTHER )
#define IS_CISCO(x)		( (x & FL_VENDOR) == FL_VENDOR_CISCO )
#define IS_NETSNMP(x)		( (x & FL_VENDOR) == FL_VENDOR_NETSNMP )

/*
	check if flags x means vendor x
	IS_VENDOR(flags,FL_VENDOR_CISCO)
*/
#define	IS_VENDOR(x,y)		( (x & FL_VENDOR) == y )

#define	DO_GETNEXT	1
#define	DO_GET		2

struct vMIB
{
	const char	*mib;
	u_8		vendor;
	u_8		type;	// get or getnext
};

vMIB	vmib[]={
	{"IF-MIB::ifDescr",				FL_VENDOR_ALL,		DO_GETNEXT},	// 0 - if description
	{"IF-MIB::ifInOctets",				FL_VENDOR_ALL,		DO_GETNEXT},	// 1 - if in octets
	{"IF-MIB::ifOutOctets",				FL_VENDOR_ALL,		DO_GETNEXT},	// 2 - if out octets
	{"IF-MIB::ifSpeed",				FL_VENDOR_ALL,		DO_GETNEXT},	// 3 - if speed
	{"IF-MIB::ifAlias",				FL_VENDOR_ALL,		DO_GETNEXT},	// 4 - if alias
	{"SNMPv2-MIB::sysUpTime.0",			FL_VENDOR_ALL,		DO_GET},	// 5 - system uptime
	{"SNMPv2-SMI::enterprises.9.2.1.56.0",		FL_VENDOR_CISCO,	DO_GET},	// 6 - cisco - cpu 5 seconds
	{"SNMPv2-SMI::enterprises.9.2.1.57.0",		FL_VENDOR_CISCO,	DO_GET},	// 7 - cisco - cpu 1 minute
	{"SNMPv2-SMI::enterprises.9.2.1.58.0",		FL_VENDOR_CISCO,	DO_GET},	// 8 - cisco - cpu 5 minutes
	{"SNMPv2-SMI::enterprises.9.9.48.1.1.1.5.1",	FL_VENDOR_CISCO,	DO_GET},	// 9 - cisco - mem used
	{"SNMPv2-SMI::enterprises.9.9.48.1.1.1.6.1",	FL_VENDOR_CISCO,	DO_GET},	// 10 - cisco - mem free
	{"SNMPv2-SMI::enterprises.2021.10.1.5.1",	FL_VENDOR_NETSNMP,	DO_GET},	// 11 - netsnmp - load 1 minute
	{"SNMPv2-SMI::enterprises.2021.10.1.5.2",	FL_VENDOR_NETSNMP,	DO_GET},	// 12 - netsnmp - load 5 minutes
	{"SNMPv2-SMI::enterprises.2021.10.1.5.3",	FL_VENDOR_NETSNMP,	DO_GET},	// 13 - netsnmp - load 15 minutes
	{"SNMPv2-SMI::enterprises.2021.4.5.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 14 - netsnmp - mem - TotalReal
	{"SNMPv2-SMI::enterprises.2021.4.3.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 15 - netsnmp - mem - TotalSwap
	{"SNMPv2-SMI::enterprises.2021.4.11.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 16 - netsnmp - mem - TotalFree
	{"SNMPv2-SMI::enterprises.2021.4.13.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 17 - netsnmp - mem - Shared
	{"SNMPv2-SMI::enterprises.2021.4.14.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 18 - netsnmp - mem - Buffers
	{"SNMPv2-SMI::enterprises.2021.4.15.0",		FL_VENDOR_NETSNMP,	DO_GET},	// 19 - netsnmp - mem - Cache
	{NULL,0,0}
};

KEYTABLE<DEVICE_DATA *>	devices;	// key is the IP

/*
	type:
		1: ifDescr
		2: ifAlias
*/
void	parse_string(DEVICE_DATA *pdev, variable_list *vars, int type)
{
	char		*p;
	int		n;

	if (vars->type == ASN_OCTET_STR)
	{
		p=(char *)malloc(vars->val_len+1);
		memcpy(p,vars->val.string, vars->val_len);
		p[vars->val_len]=0;

		n=vars->name[pdev->root_oid_len];

		if (type==1)
		{
			IFACE		iface;
			
			iface.index=n;
			iface.descr=p;
			pdev->pnetdev->ifaces.Add(n,iface);
			printf("ifDescr: %s %d %s\n",pdev->pnetdev->ip.p,iface.index, iface.descr.p);
		} else if (type==2)
		{
			IFACE	*piface;
			int	ifindex;
			
			ifindex=vars->name[pdev->root_oid_len];
		
			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface==NULL)
			{
				printf("Ignoring data(string) for unknown interface %d (type=%d)\n", ifindex, type);
			} else
			{
				printf("if-string: ip:%s  index:%d  type:%d  string:%s\n",pdev->pnetdev->ip.p, piface->index, type, p);
				piface->alias=p;
			}

		}

		free(p);
	}
}

/*
	type:
		1: bytes-in
		2: butes-out
		3: ifspeed
		4: uptime
		5: cisco cpu 5 seconds
		6: cisco cpu 1 minute
		7: cisco cpu 5 minuts
		8: cisco mem free
		9: cisco mem used
		10: netsnmp load 1 min
		11: netsnmp load 5 min
		12: netsnmp load 15 min
		13: netsnmp memory total real
		14: netsnmp memory total swap
		15: netsnmp memory (-)
*/
void	parse_number(DEVICE_DATA *pdev, variable_list *vars, int type)
{
	int	ifindex;
	u_32	sz;
	IFACE	*piface;
	
	printf("parse_number: vars->type=%d\n",vars->type);
	if (vars->type == ASN_COUNTER || vars->type == ASN_GAUGE || vars->type == ASN_TIMETICKS || vars->type == ASN_INTEGER)
	{
		sz=*vars->val.integer;
		
		if (type==1 || type==2 || type==3)
		{
			ifindex=vars->name[pdev->root_oid_len];

			piface=pdev->pnetdev->ifaces[ifindex];
			if (piface==NULL)
			{
				printf("Ignoring data(number) for unknown interface %d (type=%d)\n", ifindex, type);
			} else
			{
				printf("if-number: ip:%s  index:%d  type:%d  number:%u\n",pdev->pnetdev->ip.p, piface->index, type, sz);
				if (type==1)
				{
					piface->bytes_in=sz;
					piface->ts=time(NULL);
				} else if (type==2)
				{
					piface->bytes_out=sz;
					piface->ts=(1U*piface->ts+time(NULL))/2;	// For more accuracy
				} else if (type==3)
				{
					piface->speed=sz;
				}
			}
		} else if (type==4)
		{
			printf("uptime: %u\n",sz);
			pdev->pnetdev->uptime=sz;
		} else if (type==5)
		{
			printf("cisco CPU 5 seconds: %u%%\n",sz);
			pdev->pnetdev->cpu1=sz;
		} else if (type==6)
		{
			printf("cisco CPU 1 minute: %u%%\n",sz);
			pdev->pnetdev->cpu2=sz;
		} else if (type==7)
		{
			printf("cisco CPU 5 minutes: %u%%\n",sz);
			pdev->pnetdev->cpu3=sz;
		} else if (type==8)
		{
			printf("cisco mem used: %u\n",sz);
			pdev->pnetdev->mem_used=sz;
		} else if (type==9)
		{
			printf("cisco mem free: %u\n",sz);
			pdev->pnetdev->mem_free=sz;
		} else if (type==10)
		{
			printf("netsnmp load 1 minute: %u\n",sz);
			pdev->pnetdev->cpu1=sz;
		} else if (type==11)
		{
			printf("netsnmp load 5 minutes: %u\n",sz);
			pdev->pnetdev->cpu2=sz;
		} else if (type==12)
		{
			printf("netsnmp load 15 minutes: %u\n",sz);
			pdev->pnetdev->cpu3=sz;
		} else if (type==13)
		{
			printf("netsnmp mem total real: %u\n",sz);
			pdev->pnetdev->mem_free=sz;
			pdev->pnetdev->mem_used=sz;
		} else if (type==14)
		{
			pdev->pnetdev->mem_used+=sz;
			printf("netsnmp mem total swap: %u (%lld)\n",sz,pdev->pnetdev->mem_used);
		} else if (type==15)
		{
			pdev->pnetdev->mem_used-=sz;
			printf("netsnmp mem (-): %u (%lld)\n",sz,pdev->pnetdev->mem_used);
		}
	}
}

/*
	Send a GETNEXT request.
	Return: >0 ok
		0 error
*/
int	snmp_send_getnext(DEVICE_DATA *pdev)
{
	snmp_pdu	*pdu;
	int		ret=1;
	
	pdu=snmp_pdu_create(SNMP_MSG_GETNEXT);
	snmp_add_null_var(pdu,pdev->cur_oid, pdev->cur_oid_len);
	
	if (!snmp_send(pdev->psession, pdu))
	{
		snmp_perror("snmp_send");
		snmp_free_pdu(pdu);
		pdev->status=-1;
		ret=0;
	}

	return(ret);
}

/*
	Send a GET request.
	Return: >0 ok
		0 error
*/
int	snmp_send_get(DEVICE_DATA *pdev)
{
	snmp_pdu	*pdu;
	int		ret=1;
	
	pdu=snmp_pdu_create(SNMP_MSG_GET);
	snmp_add_null_var(pdu,pdev->cur_oid, pdev->cur_oid_len);
	
	if (!snmp_send(pdev->psession, pdu))
	{
		snmp_perror("snmp_send");
		snmp_free_pdu(pdu);
		pdev->status=-1;
		ret=0;
	}

	return(ret);
}

int	f_callback(int operation, struct snmp_session *sp, int reqid,
			struct snmp_pdu *pdu, void *magic)
{
	DEVICE_DATA	*pdev = (DEVICE_DATA *)magic;
	variable_list	*vars;
	int		done=0;
	
	printf("----- callback ------\n");
	printf("done=%d pdev->status=%d\n",done,pdev->status);
	
	if (pdev->status>=100 || pdev->status<0)
		return(0);
	
	printf("koko pdu=%p  pdu->errstat=%d  operation=%d\n",pdu, pdu->errstat, operation);
	if (operation != NETSNMP_CALLBACK_OP_RECEIVED_MESSAGE ||
		pdu==NULL ||
		pdu->errstat!=SNMP_ERR_NOERROR)
	{
		pdev->status=-1;
		return(1);
	}
	printf("koko2\n");
	
	/*
		done:
			0: nothing
			1: the getnext is over (oid mismatch)
			2: oid not increasing
			3: the getnext is over (no more data)
			4: it was a get request and data came
	*/
	for (vars=pdu->variables ; (vars && (done==0 || done==1 || done==2)) ; vars=vars->next_variable)
	{
/*---*/		print_variable(vars->name, vars->name_length, vars);

		if (vars->name_length < pdev->root_oid_len ||
			memcmp(vars->name, pdev->root_oid, pdev->root_oid_len * sizeof(oid)))
		{
			printf("%s(%d): done\n",pdev->pnetdev->ip.p,pdev->status);
			done=1;
//			pdev->status++;
		} else if ((snmp_oid_compare(pdev->cur_oid, pdev->cur_oid_len,
				vars->name, vars->name_length) >=0) && 
				(vmib[(pdev->status-1)/2].type==DO_GETNEXT))	// only check on getnext requests
		{
			printf("%s(%d): OID not increasing!\n", pdev->pnetdev->ip.p, pdev->status);
//			pdev->status++;
			done=2;
		} else if ((vars->type == SNMP_ENDOFMIBVIEW) ||
			(vars->type == SNMP_NOSUCHOBJECT) ||
			(vars->type == SNMP_NOSUCHINSTANCE))
		{
			done=3;
		} else
		{
			switch(pdev->status)
			{
				case 1:	// if descr
					parse_string(pdev, vars, 1); break;
				case 3:	// if in octets
					parse_number(pdev, vars, 1); break;
				case 5:	// if out octets
					parse_number(pdev, vars, 2); break;
				case 7:	// if speed
					parse_number(pdev, vars, 3); break;
				case 9:	// if alias
					parse_string(pdev, vars, 2); break;
				case 11:	// uptime
					parse_number(pdev, vars, 4); done=4; break;
				case 13:	// cisco cpu 5 seconds
					parse_number(pdev, vars, 5); done=4; break;
				case 15:	// cisco cpu 1 minute
					parse_number(pdev, vars, 6); done=4; break;
				case 17:	// cisco cpu 5 minutes
					parse_number(pdev, vars, 7); done=4; break;
				case 19:	// cisco mem used
					parse_number(pdev, vars, 8); done=4; break;
				case 21:	// cisco mem free
					parse_number(pdev, vars, 9); done=4; break;
				case 23:	// netsnmp load 1 minute
					parse_number(pdev, vars, 10); done=4; break;
				case 25:	// netsnmp load 5 minutes
					parse_number(pdev, vars, 11); done=4; break;
				case 27:	// netsnmp load 15 minutes
					parse_number(pdev, vars, 12); done=4; break;
				case 29:	// netsnmp mem total real
					parse_number(pdev, vars, 13); done=4; break;
				case 31:	// netsnmp mem total swap
					parse_number(pdev, vars, 14); done=4; break;
				case 33:	// netsnmp mem total free
				case 35:	// netsnmp mem shared
				case 37:	// netsnmp mem buffers
				case 39:	// netsnmp mem cache
					parse_number(pdev, vars, 15); done=4; break;
			}
			memcpy((char*)pdev->cur_oid, (char *)vars->name, vars->name_length*sizeof(oid));
			pdev->cur_oid_len=vars->name_length;
		}
		
	}
	if (done==1 || done==2 || done==3 || done==4)
	{
		pdev->status++;
	} else
	{
		if (!snmp_send_getnext(pdev))
			pdev->status=-1;
	}
//	if (done==0 && pdev->status!=-1)
//		if (!snmp_send_getnext(pdev))
//			pdev->status=-1;
	printf("done=%d pdev->status=%d\n",done,pdev->status);
	printf("----- end of callback ------\n\n");
	
	return(1);
}

void	snmp_init_queries()
{
	LONGLINE	*pll;
	NETDEV		*pnetdev;
	DEVICE_DATA	*pdev;
	
	netdevs.Rewind();
	while ((pll=netdevs.Get_Next())!=NULL)
	{
		pnetdev=netdevs[*pll];
		pdev=new DEVICE_DATA;
		
		devices.Add(pll->p, pdev);
		
		pdev->pnetdev=pnetdev;
		pdev->status=0;
		
		printf("init for: %s %s\n",pnetdev->ip.p, pnetdev->community.p);
		
		snmp_sess_init(&pdev->session);
		pdev->session.version=SNMP_VERSION_1;
		pdev->session.peername=pdev->pnetdev->ip.p;
		pdev->session.community=(u_char *)pdev->pnetdev->community.p;
		pdev->session.community_len=pdev->pnetdev->community.len;
		pdev->session.callback=f_callback;
		pdev->session.callback_magic=pdev;
		
		pdev->psession=snmp_open(&pdev->session);
		if (pdev->psession==NULL)
		{
			pdev->status=-1;
			snmp_perror("snmp_open");
		}
	}
}

void	snmp_send_requests()
{
	DEVICE_DATA	*pdev;
	LONGLINE	*pll;
	oid		*ret;
	const char	*mib;
	int		ok,idx;
	
	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll]);
		if (pdev->status>=0 && pdev->status<100)
		{
			if (pdev->status%2 == 0)
			{
				idx=pdev->status/2;
				
				ok=0;
				
				while (!ok)	// find next request to send
				{
					if (vmib[idx].mib==NULL)	// end of requests
					{
						ok=1;
					} else
					{
						printf("flags: %d\n",pdev->pnetdev->flags);
						printf("vendor: %d\n",vmib[idx].vendor);
						if ((vmib[idx].vendor==FL_VENDOR_ALL) ||
							((pdev->pnetdev->flags & FL_VENDOR) == vmib[idx].vendor))
//							(((pdev->pnetdev->flags & 0x0f) == 0x01) && (IS_CISCO(vmib[idx].vendor))))
						{
							ok=2;
						} else
						{
							idx++;
							pdev->status+=2;
						}
					}
				}
				if (ok==1)
				{
					pdev->status=100;	// the end
				} else
				{
					mib=vmib[idx].mib;
				
					pdev->root_oid_len=MAX_OID_LEN;
				
					printf("mib=%s\n",mib);
					ret=snmp_parse_oid(mib,
						pdev->root_oid,
						&pdev->root_oid_len);
					if (!ret)
					{
						snmp_perror("snmp_parse_oid");
						pdev->status=-1;
					} else
					{
					
						memcpy(pdev->cur_oid, pdev->root_oid, sizeof(oid) * pdev->root_oid_len);
						pdev->cur_oid_len=pdev->root_oid_len;
					
						pdev->status++;

						switch(vmib[idx].type)
						{
							case DO_GETNEXT:
								snmp_send_getnext(pdev);
								break;
							case DO_GET:
								snmp_send_get(pdev);
								break;
						}
					}
				}
			}
		}
		
		if (pdev->status<0 || pdev->status==100)
		{
			snmp_close(pdev->psession);
		}
		
	}
}

int	snmp_devices_done()
{
	DEVICE_DATA	*pdev;
	LONGLINE	*pll;
	
	devices.Rewind();
	while ((pll=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll]);
		if (pdev->status>=0 && pdev->status<100)
			return(0);
	}
	
	return(1);
}

void	snmp_do_select()
{
	fd_set		fdr;
	int		fds;
	int		block;
	struct timeval	tv;
	int		ret;

	snmp_send_requests();
        
	while (!snmp_devices_done())
	{
		block=1;
		fds=0;
		FD_ZERO(&fdr);

		snmp_select_info(&fds, &fdr, &tv, &block);
		ret=select(fds, &fdr, NULL, NULL, block ? NULL : &tv);
		if (ret)
			snmp_read(&fdr);
		else
			snmp_timeout();
		snmp_send_requests();
	}
}

void	snmp_doit()
{
	init_snmp("V-SNMP");
	
	snmp_init_queries();
	snmp_do_select();
}

void	update_db()
{
	DEVICE_DATA	*pdev;
	NETDEV		*pnetdev;
	IFACE		*piface;
	LONGLINE	*pll1;
	u_32		key;

	db_begin_update();
	do_rotate();
	devices.Rewind();
	while ((pll1=devices.Get_Next())!=NULL)
	{
		pdev=*(devices[*pll1]);
		
		pnetdev=pdev->pnetdev;

		update_netdev(pnetdev->name.p, pnetdev->uptime,
				pnetdev->cpu1, pnetdev->cpu2, pnetdev->cpu3,
				pnetdev->mem_used, pnetdev->mem_free);
		update_memacct(pnetdev->ip.p, pnetdev->mem_used);
		update_cpuacct(pnetdev->ip.p, pnetdev->cpu2);
				
		pnetdev->ifaces.Rewind();
		
		while ((key=pnetdev->ifaces.Get_Next())!=NK_INVALID)
		{
			piface=pnetdev->ifaces[key];

			update_iface(pnetdev->ip.p, piface);

			// if uptime is less than the update interval
			if ( (pnetdev->uptime/100) < (u_32)(piface->ts - piface->last_ts)) 
			{
				printf("Detected restart for %s (%d - %d - %d)\n",pnetdev->name.p,
						pnetdev->uptime/100, piface->ts, piface->last_ts);
				piface->last_bytes_in=piface->bytes_in+1;
				piface->last_bytes_out=piface->bytes_out+1;
			}
//			update_iface(pnetdev->ip.p, piface->descr.p,
//				piface->alias.p, piface->index,
//				piface->bytes_in, piface->bytes_out,
//				piface->speed, piface->ts);

			update_ifacct(pnetdev->ip.p, piface);

//			update_ifacct(pnetdev->ip.p, piface->descr.p,
//				piface->bytes_in, piface->bytes_out, piface->ts);
//			update_ifacct(pnetdev->ip.p, piface->descr.p,
//				 piface->bytes_in, piface->bytes_out,
//				 ,,
//				 piface->speed,
//				 . piface->ts);
		}
	}
	db_end_update();
	do_final();
}
