/*
  Copyright (c) 2002 Stefanos Harhalakis

  This file is part of proclist.

  proclist is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  proclist is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with proclist; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  $Id: test.cc,v 1.4 2004/06/13 13:34:20 v13 Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <grp.h>
#include <pwd.h>
#include <ctype.h>

char	**buf=NULL;		// the buffer
int	lines=0;		// the number of lines
//int	pstype=-1;
char	*ps=NULL;

struct ps_line
{
	long	pid;
	long	ppid;
	long	euid;
	long	egid;
	char	*tty;
	char	*cmd;
};

#define FLD_NO	6	// number of fields in a line

ps_line		**ps_list=NULL;
int		ps_lines=0;

void	clear_buffer()
{
	int	n;

	if (buf==NULL)
		return;

	for (n=0;n<lines;n++)
	{
		free(buf[n]);
	}
	free(buf);
	lines=0;
	buf=NULL;
}

void	fail()
{
//	printf("error\n");
	exit(1);
}

void	add_buf_line(const char *line)
{
	lines++;
	if (buf==NULL)
		buf=(char **)malloc(sizeof(char *) * lines);
	else
		buf=(char **)realloc(buf, sizeof(char *) * lines);

	if (buf==NULL)
		fail();

	buf[lines-1]=strdup(line);
}

void	clear_ps_list()
{
	int	n;

	if (ps_list==NULL)
		return;

	for (n=0;n<ps_lines;n++)
	{
		free(ps_list[n]->tty);
		free(ps_list[n]->cmd);
		free(ps_list[n]);
	}
	free(ps_list);
	ps_lines=0;
	ps_list=NULL;
}

void	add_ps_line(long pid, long ppid, long euid, long egid, char *tty, char *cmd)
{
	ps_lines++;
	if (ps_list==NULL)
		ps_list=(ps_line **)malloc(sizeof(ps_line *) * ps_lines);
	else
		ps_list=(ps_line **)realloc(ps_list, sizeof(ps_line *) * ps_lines);

	if (ps_list==NULL)
		fail();

	ps_list[ps_lines-1]=(ps_line *)malloc(sizeof(ps_line));
	ps_list[ps_lines-1]->pid=pid;
	ps_list[ps_lines-1]->ppid=ppid;
	ps_list[ps_lines-1]->euid=euid;
	ps_list[ps_lines-1]->egid=egid;
	ps_list[ps_lines-1]->tty=strdup(tty);
	ps_list[ps_lines-1]->cmd=strdup(cmd);
}

uid_t	get_uid(const char *name)
{
	uid_t	id;
	char	*p;
	passwd	*pwd;

	id=(uid_t)strtol(name,&p,10);

	if ((p==NULL) || (*p==0))
		return(id);

	pwd=getpwnam(name);
	if (pwd==NULL)
		return((uid_t)-1);

	id=pwd->pw_uid;

	return(id);
}

gid_t	get_gid(const char *name)
{
	gid_t	id;
	char	*p;
	group	*grp;

	id=(gid_t)strtol(name,&p,10);

	if ((p==NULL) || (*p==0))
		return(id);

	grp=getgrnam(name);
	if (grp==NULL)
		return((gid_t)-1);

	id=grp->gr_gid;

	return(id);
}
void	parse_add_ps_line(char *line)
{
	char		*p;
	char		buf[101];
	int		n;
	long		pid,ppid,euid,egid;
	char		*p2;

#define	NOSPACE	while(*p==' ' || *p=='\t') p++;
#define GET_STR	n=0; \
		while((n<100) && (*p!=' ' && *p!='\t' && *p!=0)) \
		{	\
			buf[n]=*p;	\
			n++; p++;	\
		}	\
		buf[n]=0;

	p=line;

	NOSPACE
	GET_STR
	pid=strtol(buf,&p2,10);
	if ((p2!=NULL) && (*p2!=0))	// invalid line
		return;

//	printf("pid={%s} (%d)\n",buf,pid);

	NOSPACE
	GET_STR
	ppid=strtol(buf,&p2,10);
	if ((p2!=NULL) && (*p2!=0))	// invalid line
		return;

//	printf("ppid={%s} (%d)\n",buf,ppid);

	NOSPACE
	GET_STR
	euid=get_uid(buf);

//	printf("user={%s} (%d)\n",buf,euid);

	NOSPACE
	GET_STR
	egid=get_gid(buf);

	NOSPACE
	GET_STR
	// buf has the tty
	
	NOSPACE
	// p is the command

//	printf("group={%s} (%d)\n",buf,egid);

	add_ps_line(pid,ppid,euid,egid,buf,p);

#undef NOSPACE
}

/*
	Do the ps.
	Return the exit status of the command.
*/
int	do_ps(const char *cmd)
{
	FILE	*f;
	char	line[1024],*p;
	int	n;
	int	ret;

	clear_buffer();

	f=popen(cmd,"r");

	if (f==NULL)
		fail();

	while (!feof(f))
	{
		p=fgets(line, 1024, f);

		if (p!=NULL)
		{
			if (strlen(line)>1020)	// WTF?
				return(1);

			add_buf_line(line);
		}
	}

	n=pclose(f);

	return(n);
}

/*
	Perform a check to the list
	return: 1: OK, 0: Errors
*/
int	check_ps_list()
{
	int	n,m;
	int	OK;
	ps_line	*p;
	int	found_init=0;
	int	found_self=0;
	pid_t	mypid=getpid();
	char	*pp;

	n=0;
//	printf("ps_lines=%d mypid=%d\n",ps_lines,mypid);
	while (n<ps_lines)
	{
		p=ps_list[n];
//		printf("%s",buf[n+1]);
//		printf("%d %d %d %d %s %s\n",p->pid, p->ppid, p->euid, p->egid, p->tty, p->cmd);
#define DBG	fprintf(stderr,"%s",buf[n+1]);	\
		fprintf(stderr,"%d %d %d %d %s %s\n",p->pid, p->ppid, p->euid, p->egid, p->tty, p->cmd);
		if (p->pid==1)	// this is the init
		{
			if (found_init)	// if it was found again it is an error
			{
				fprintf(stderr,"Init found twice\n");
				DBG
				return(0);
			}

//			printf("%s",buf[n+1]);
//			printf("found init: %d %d %d %d\n",p->pid, p->ppid, p->euid, p->egid);
			found_init=1;
		} else if (p->pid==mypid)	// self
		{
			if (found_self)
			{
				fprintf(stderr,"Self found twice\n");
				DBG
				return(0);
			}

//			printf("%s",buf[n+1]);
//			printf("found self: %d %d %d %d\n",p->pid, p->ppid, p->euid, p->egid);

			found_self=1;

			if (p->ppid!=getppid())	// invalid
			{
				fprintf(stderr,"ppid invalid for self\n");
				DBG
				return(0);
			}
			if (p->euid!=geteuid())
			{
				fprintf(stderr,"euid invalid for self\n");
				DBG
				return(0);
			}

#ifdef FREEBSD
			if (p->euid!=getgid())
#else
			if (p->egid!=getegid())
#endif
			{
#ifdef FREEBSD
				fprintf(stderr,"rgid invlalid for self\n");
#else
				fprintf(stderr,"egid invlalid for self\n");
#endif
				DBG
				return(0);
			}
		}

		pp=strchr(p->cmd,' ');	// there should be no space in command
		if (pp!=NULL)
		{
			while (*pp==' ')
				pp++;
			if (*pp!=0 && *pp!='\n' && *pp!='\r')	// unless it is at the end
			{
				fprintf(stderr,"Spaces in command field\n");
				DBG
				return(0);
			}
		}
		pp=strchr(p->cmd,'/'); // no path
		if (pp!=NULL && p->euid==0)	// Check for kernel threads
		{
			pp++;
			while (isdigit(*pp)) pp++;
		}
		if (pp!=NULL && *pp!=0 && *pp!='\n' && *pp!='\r')
		{
			fprintf(stderr,"/ in command field\n");
			DBG
			return(0);
		}

		n++;
	}

	if (found_self && found_init)
		return(1);

	return(0);
}

/*	Check a ps
	Return 0: OK, other: failed
*/
int	check_ps()
{
	char	cmd[1024];
	int	n;

//	sprintf(cmd,"%s %s 2>/dev/null",ps,pstest[pstype]);
//	sprintf(cmd,"%s 2>/dev/null",ps);
	sprintf(cmd,"%s",ps);
	n=do_ps(cmd);

	if (n!=0)
		return(1);

//	printf("\"%s\" runs\n",cmd);fflush(stdout);

	clear_ps_list();

	for (n=0;n<lines;n++)
	{
//		printf("%s",buf[n]);
		parse_add_ps_line(buf[n]);
	}

	n=check_ps_list();

/*	if (n)
	{
		printf("\"%s\" works\n",cmd);fflush(stdout);
	}*/

	return( ( n==0 ? 1 : 0) );
}

void	find_ps()
{
	int	n;

//	pstype=0;
//	n=1;
//	while ((n!=0) && (pstest[pstype]!=NULL))
//	{
		n=check_ps();
//		if (n!=0)
//			pstype++;
//	}

	if (n!=0)
		fail();
}

/*void	do_print()
{
	printf("%s\n",pstest[pstype]);
}*/

/*void	set_gid()
{
	gid_t	groups[100];
	int	n,m;
	gid_t	egid;

	egid=getegid();

	n=getgroups(100,groups);
	for (m=0;m<n;m++)
	{
		if (groups[m]!=egid)
		{
			n=setegid(groups[m]);
			if (n<0)
			{
				fprintf(stderr,"Error changing gid\n");
			}
			return;
		}
	}
}*/

int main(int argn, char *argc[])
{
	if(argn==1)
	{
//		ps="ps";
		printf("Pass ps command as first argument\n");
		fail();
	} else
	{
		ps=argc[1];
	}

//	set_gid();

	find_ps();
//	printf("Result is: %s\n",pstest[pstype]);

//	do_print();

	exit(0);
}
